use actix_cors::Cors;
use actix_web::ResponseError;
use actix_web::{get, web, App, HttpResponse, HttpServer, Responder};
use anyhow::{Context, Result};
use chrono::{DateTime, Datelike, NaiveTime};
use chrono::{Local, NaiveDate};
use graphql_client::{GraphQLQuery, Response};
use latlon::Point;
use plan_query::{PlanQueryPlanItinerariesLegsRoute, Variables};
use reqwest::Client;
use serde::{Deserialize, Serialize};
use std::error::Error;
use std::fmt::{self};
use std::vec;
use tokio::sync::Semaphore;

use std::sync::Arc;
use std::time::{Duration, Instant};
#[derive(Clone)]
struct AppState {
    client: Arc<Client>,
    semaphore: Arc<Semaphore>,
}
struct Stopwatch {
    start_time: Instant,
    lap_time: Instant,
    laps: Vec<Duration>,
}

impl Stopwatch {
    fn new() -> Self {
        let now = Instant::now();
        Stopwatch {
            start_time: now,
            lap_time: now,
            laps: Vec::new(),
        }
    }

    fn lap(&mut self) {
        let now = Instant::now();
        let lap = now.duration_since(self.lap_time);
        self.laps.push(lap);
        self.lap_time = now;
    }

    fn total_elapsed(&self) -> Duration {
        Instant::now().duration_since(self.start_time)
    }

    fn print_results(&self) {
        for (i, lap) in self.laps.iter().enumerate() {
            println!("Operation {} took: {:?}", i + 1, lap);
        }
        println!("Total time: {:?}", self.total_elapsed());
    }
}

type OffsetDateTime = String;

#[derive(GraphQLQuery, Debug)]
#[graphql(
    schema_path = "/home/adam/projects/OpenTripPlanner/src/main/resources/org/opentripplanner/apis/gtfs/schema.graphqls",
    query_path = "plan_query.graphql",
    response_derives = "Debug"
)]
pub struct PlanQuery;

#[get("/helloworld")]
async fn hello_world() -> impl Responder {
    HttpResponse::Ok()
        .content_type("text/html")
        // 111B
        .body("<style>*{text-align:center;align-content:center;height:100%;margin:0;color-scheme:dark</style><h1>Hello, World!")
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "kebab-case")]
struct Info {
    from: String,
    to: String,
    date: Option<String>,
    time: Option<String>,
    max_transfers: Option<i64>,
    min_time_transfers: Option<i64>,
    max_time_transfers: Option<i64>,
    max_walk_time: Option<i64>,
}

#[derive(Deserialize, Debug, Serialize, Clone)]
struct Itinerary {
    from: String,
    to: String,
    start_time: String,
    end_time: String,
    route: String,
}

// Define a wrapper error type for Actix
#[derive(Debug)]
struct AppError(anyhow::Error);

impl fmt::Display for AppError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.0)
    }
}

impl ResponseError for AppError {
    fn error_response(&self) -> HttpResponse {
        HttpResponse::BadRequest().json(format!("Error: {}", self.0))
    }
}

impl From<anyhow::Error> for AppError {
    fn from(err: anyhow::Error) -> AppError {
        AppError(err)
    }
}

#[get("/plan")]
async fn plan(
    state: web::Data<AppState>,
    parameters: web::Query<Info>,
) -> Result<web::Json<Vec<Vec<Itinerary>>>, AppError> {

    let from_coordinate = parse_coordinate(&parameters.from)
        .with_context(|| format!("Failed to parse 'from' coordinate: '{}'", parameters.from))?;

    let to_coordinate = parse_coordinate(&parameters.to)
        .with_context(|| format!("Failed to parse 'to' coordinate: '{}'", parameters.to))?;


    let date = match &parameters.date {
        Some(v) if !v.trim().is_empty() => Some(convert_date(v)?),
        _ => None,
    };

    let time = match &parameters.time {
        Some(v) if !v.trim().is_empty() => Some(convert_time(v)?),
        _ => None,
    };

    let variables: Variables = plan_query::Variables {
        from: Some(plan_query::InputCoordinates {
            lat: from_coordinate.y(),
            lon: from_coordinate.x(),
            address: None,
            location_slack: None,
        }),
        to: Some(plan_query::InputCoordinates {
            lat: to_coordinate.y(),
            lon: to_coordinate.x(),
            address: None,
            location_slack: None,
        }),
        date,
        time,
        max_transfers: parameters.max_transfers,
        min_transfer_time: parameters.min_time_transfers,
    };

    let request_body = PlanQuery::build_query(variables);

    let res = state
        .client
        .post("http://127.0.0.1:8080/otp/gtfs/v1")
        .json(&request_body)
        .send()
        .await
        .map_err(|e| AppError(e.into()))?;

    let response_body: Response<plan_query::ResponseData> =
        res.json().await.map_err(|e| AppError(e.into()))?;

    let mut output: Vec<Vec<Itinerary>> = Vec::new();
    println!("{:#?}", response_body);
    for itinerary in response_body.data.unwrap().plan.unwrap().itineraries {
        let mut temp_output_json = vec![];
        for leg in itinerary.unwrap().legs {
            let leg = leg.unwrap();
            let test = leg
                .route
                .unwrap_or(PlanQueryPlanItinerariesLegsRoute {
                    gtfs_id: String::new(),
                    long_name: Some(String::new()),
                    short_name: Some("655WALK".to_string()),
                })
                .short_name;
            let var_name = test.clone().unwrap();
            let test2 = var_name.strip_prefix("655").unwrap_or(&var_name);
            dbg!(test);
            dbg!(&var_name);
            dbg!(test2);

            temp_output_json.push(Itinerary {
                from: leg.from.name.unwrap(),
                to: leg.to.name.unwrap(),
                start_time: leg
                    .from
                    .departure
                    .unwrap()
                    .scheduled_time
                    .split('T')
                    .last()
                    .unwrap()
                    .to_owned(),
                end_time: leg
                    .to
                    .departure
                    .unwrap()
                    .scheduled_time
                    .split('T')
                    .last()
                    .unwrap()
                    .to_owned(),
                route: test2.to_owned(),
            });

        }
        output.push(temp_output_json);
    }
    if parameters.max_time_transfers.is_some() {
        output = output
            .iter()
            .filter(|x| {
                let mut last_time = "";
                for ii in x.iter() {
                    if ii.route != "WALK" {
                        if !last_time.is_empty() {
                            if get_time_difference_in_minutes(last_time, &ii.start_time).unwrap()
                                > parameters.max_time_transfers.unwrap()
                            {
                                return false;
                            }
                        }
                        last_time = &ii.end_time;
                    }
                }

                return true;
            })
            .cloned()
            .collect();
    }

    if parameters.max_walk_time.is_some() {
        output = output
            .iter()
            .filter(|x| {
                for ii in x.iter() {
                    if ii.route == "WALK" {
                        if get_time_difference_in_minutes(&ii.start_time, &ii.end_time).unwrap()
                            > parameters.max_walk_time.unwrap()
                        {
                            return false;
                        }
                    }
                }

                return true;
            })
            .cloned()
            .collect();
    }

    dbg!(&output);
    Ok(web::Json(output))
}

#[actix_web::main]
async fn main() -> std::io::Result<()> {
    let client = Client::builder()
        // .pool_max_idle_per_host(100)
        // .pool_idle_timeout(std::time::Duration::from_secs(90))
        .build()
        .expect("Failed to create HTTP client");

    let app_state = web::Data::new(AppState {
        client: Arc::new(client),
        semaphore: Arc::new(Semaphore::new(100)), // Limit concurrent requests to 50
    });
    HttpServer::new(move || {
        App::new()
            .wrap(
                Cors::default() // Allow all origins
                    .allow_any_origin()
                    .allow_any_method()
                    .allow_any_header(),
            )
            .app_data(app_state.clone())
            .service(hello_world)
            .service(plan)
    })
    // .workers(12 * 4) // Adjust based on your CPU cores
    // .backlog(8192)
    // .max_connections(100_000)
    // .max_connection_rate(10_000)
    // .keep_alive(Duration::from_secs(120))
    // .client_request_timeout(Duration::from_secs(60))
    .bind(("127.0.0.1", 8081))?
    .run()
    .await
}

use vyhledavac_spojeni_api::optimize_coordinate_3;

fn parse_coordinate(input: &str) -> Result<Point<f64>> {
    let mut coordinate = optimize_coordinate_3(input);

    if coordinate.split(",").collect::<Vec<&str>>().len() == 4 {
        coordinate = format_coordinate(&coordinate);
    }
    if !coordinate.contains(['°', '"', '′', '.', '″']) {
        coordinate = format_coordinates(&coordinate)?;
    }

    match latlon::parse(&coordinate) {
        Ok(v) => Ok(v),
        Err(_) => {
            anyhow::bail!("Failed to parse coordinate: {}", coordinate);
        }
    }
}
fn format_coordinate(input: &str) -> String {
    let mut count = 0;
    input
        .chars()
        .map(|c| {
            if c == ',' {
                count += 1;
                if count == 1 || count == 3 {
                    '.'
                } else {
                    ','
                }
            } else {
                c
            }
        })
        .collect()
}

fn format_coordinates(input: &str) -> Result<String> {
    let mut parts: Vec<&str> = input.split_whitespace().collect();
    if parts.len() != 2 {
        parts = split_after_first_letter(input)?;
    }

    let lat = format_part(parts[0])?;
    let lon = format_part(parts[1])?;

    Ok(format!("{lat} {lon}"))
}

fn format_part(part: &str) -> Result<String> {
    let coords = part;
    if coords.len() < 9 {
        anyhow::bail!("Coordinate part is too short");
    }
    let milliseconds_direction = &coords[coords.len() - 4..];
    let seconds = &coords[coords.len() - 6..coords.len() - 4];
    let minutes = &coords[coords.len() - 8..coords.len() - 6];
    let degrees = &coords[..coords.len() - 8];

    Ok(format!(
        "{degrees} {minutes} {seconds}.{milliseconds_direction}"
    ))
}

fn split_after_first_letter(s: &str) -> Result<Vec<&str>> {
    if let Some(index) = s.find(char::is_alphabetic) {
        let (start, rest) = s.split_at(index + 1);
        Ok(vec![start, rest])
    } else {
        anyhow::bail!("no letter found in coordinate");
    }
}

fn convert_date(date_str: &str) -> Result<String> {
    let formats_with_year = [
        "%d.%m.%Y", "%Y-%m-%d", "%d/%m/%Y", "%Y/%m/%d", "%d-%m-%Y", "%Y.%m.%d", "%d %m %Y",
        "%Y %m %d",
    ];
    let formats_without_year = ["%d.%m", "%d-%m", "%d/%m", "%d %m"];

    for format in formats_with_year {
        if let Ok(date) = NaiveDate::parse_from_str(date_str, format) {
            return Ok(date.format("%Y-%m-%d").to_string());
        }
    }

    let current_year = Local::now().year();
    for format in formats_without_year {
        if let Ok(date) = NaiveDate::parse_from_str(
            &format!("{}.{}", date_str, current_year),
            &format!("{}.%Y", format),
        ) {
            return Ok(date.format("%Y-%m-%d").to_string());
        }
    }

    anyhow::bail!("Failed to convert date")
}

fn convert_time(time_str: &str) -> Result<String> {
    let formats = ["%H:%M:%S", "%H:%M", "%H %M %S", "%H %M"];

    for format in formats {
        if let Ok(time) = NaiveTime::parse_from_str(time_str, format) {
            return Ok(time.format("%H:%M:%S").to_string());
        }
    }

    if let Ok(hour) = time_str.parse::<u32>() {
        if hour < 24 {
            return Ok(format!("{hour}:00:00"));
        }
    }

    anyhow::bail!("Failed to convert time")
}

fn get_time_difference_in_minutes(time1: &str, time2: &str) -> Result<i64, Box<dyn Error>> {
    let date_prefix = "2023-01-01T";

    let time1 = DateTime::parse_from_rfc3339(&format!("{}{}", date_prefix, time1))?;
    let time2 = DateTime::parse_from_rfc3339(&format!("{}{}", date_prefix, time2))?;

    let difference = time2.signed_duration_since(time1);
    Ok(difference.num_minutes())
}

#[cfg(test)]
mod tests {
    use vyhledavac_spojeni_api::COORDINATES;

    use super::*;

    #[test]
    fn test_parse_coordinate() {
        for coordinate in COORDINATES {
            if let Err(_) = parse_coordinate(coordinate) {
                panic!("failed to parse coordinate: {coordinate}")
            }
        }
    }
}
