use core::str;
use std::{
    env, fs::File, io::Read, path::{Path, PathBuf}, process, sync::atomic::{AtomicUsize, Ordering}
};

use encoding_rs::WINDOWS_1250;
use encoding_rs_rw::DecodingReader;
use walkdir::WalkDir;

use proj::Proj;

#[allow(non_snake_case)]
#[allow(dead_code)]
#[derive(Debug, serde::Deserialize)]
struct Record {
    Kód_ADM: String,
    Kód_obce: String,
    Název_obce: String,
    Kód_MOMC: String,
    Název_MOMC: String,
    Kód_obvodu_Prahy: String,
    Název_obvodu_Prahy: String,
    Kód_části_obce: String,
    Název_části_obce: String,
    Kód_ulice: String,
    Název_ulice: String,
    Typ_SO: String,
    Číslo_domovní: String,
    Číslo_orientační: String,
    Znak_čísla_orientačního: String,
    PSČ: String,
    Souřadnice_Y: String,
    Souřadnice_X: String,
    Platí_Od: String,
}

fn main() {
    let args: Vec<String> = env::args().collect();
    if args.len() != 2 {
        eprintln!("Usage: {} <folder_path>", args[0]);
        process::exit(1);
    }
    let folder_path = &args[1];
    if !Path::new(folder_path).is_dir() {
        eprintln!("Error: Folder not found: {folder_path}");
        process::exit(1);
    }
    let files: Vec<PathBuf> = WalkDir::new(folder_path)
        .into_iter()
        .filter_map(std::result::Result::ok)
        .filter(|e| e.file_type().is_file() && e.metadata().map(|m| m.len() > 0).unwrap())
        .map(|e| e.path().to_owned())
        .collect();


    let from = "EPSG:5514";
    let to = "EPSG:4326";

    let to_wgs84 = Proj::new_known_crs(from, to, None).unwrap();

    let number_of_errors = AtomicUsize::new(0);

    for file in files {
        let mut fh = File::open(file.clone()).unwrap();
        let mut reader: Vec<u8> = Vec::new();
        fh.read_to_end(&mut reader).expect("Could not read file");

        let decoding_reader = DecodingReader::new(&*reader, WINDOWS_1250.new_decoder());

        let mut rdr = csv::ReaderBuilder::new()
            .delimiter(b';')
            .from_reader(decoding_reader);
        let abc = rdr
            .byte_headers()
            .unwrap()
            .iter()
            .map(|x| str::from_utf8(x).unwrap().replace(' ', "_"))
            .collect();
        rdr.set_headers(abc);
        for result in rdr.deserialize() {
            let record: Record = result.unwrap();
            // Název ulice / Název části obce
            // Typ SO
            // Číslo domovní
            // /
            // Číslo orientační

            // Název části obce

            // PSČ
            // Název obce | Název obvodu Prahy
            let street = if record.Název_ulice.is_empty() {
                if record.Název_části_obce == record.Název_obce {
                    ""
                } else {
                    &record.Název_části_obce
                }
            } else {
                &record.Název_ulice
            };
            let first_line = format!(
                "{} {} {}{}",
                street,
                if record.Typ_SO == "č.p." && !street.is_empty() {
                    ""
                } else {
                    &record.Typ_SO
                },
                record.Číslo_domovní,
                if record.Číslo_orientační.is_empty() {
                    String::new()
                } else {
                    "/".to_owned() + record.Číslo_orientační.as_str() + record.Znak_čísla_orientačního.as_str()
                }
            );
            let third_line = format!(
                "{} {}",
                record.PSČ,
                if record.Název_obvodu_Prahy.is_empty() {
                    &record.Název_obce
                } else {
                    &record.Název_obvodu_Prahy
                }
            );
            let mut address = if (record.Název_ulice.is_empty()
                && street == record.Název_části_obce)
                || (record.Název_části_obce == record.Název_obce)
            {
                format!("{first_line}, {third_line}")
            } else {
                format!("{first_line}, {}, {third_line}", record.Název_části_obce)
            };
            address = remove_excess_whitespace(address.trim());

            if record.Souřadnice_Y.is_empty() {
                number_of_errors.fetch_add(1, Ordering::Relaxed);
                continue;
            }
            let result = to_wgs84
                .convert((
                    -record.Souřadnice_Y.parse::<f64>().unwrap(),
                    -record.Souřadnice_X.parse::<f64>().unwrap(),
                ))
                .unwrap();
            println!("{address};{};{}", format_float(result.1), format_float(result.0));
        }
    }

    if number_of_errors.load(Ordering::Relaxed) == 0 {
        return;
    }

    eprintln!(
        "Completed but {} addresses didn't have gps coordinates so they were discarded",
        number_of_errors.load(Ordering::Relaxed)
    );
}

fn remove_excess_whitespace(s: &str) -> String {
    let mut result = String::new();
    let mut last_was_space = false;

    for c in s.chars() {
        if c.is_whitespace() {
            if !last_was_space {
                result.push(' ');
                last_was_space = true;
            }
        } else {
            result.push(c);
            last_was_space = false;
        }
    }

    result.trim().to_string()
}

fn format_float(num: f64) -> String {
    let mut formatted = format!("{num:.6}");
    if !formatted.contains('.') {
        return formatted;
    }
    while formatted.ends_with('0') {
        formatted.pop();
    }
    if formatted.ends_with('.') {
        formatted.pop();
    }
    formatted
}
