use criterion::{black_box, criterion_group, criterion_main, Criterion};
use rand::Rng;
use std::time::Duration;

// Original version
fn parse_time_original(time_str: String) -> u32 {
    let parts: Vec<&str> = time_str.split(':').collect();
    if parts.len() != 3 {
        panic!("Invalid time format");
    }
    // let hours: u32 = parts[0].parse().expect(&format!("{time_str:#?}"));
    // let hours: u32 = parts[0].parse().unwrap();
    let hours: u32 = parts[0].parse().unwrap_or_else(|_| panic!("Failed to parse time: {time_str:#?}"));
    let minutes: u32 = parts[1].parse().unwrap();
    let seconds: u32 = parts[2].parse().unwrap();
    assert!(seconds == 0);
    (hours * 3600 + minutes * 60 + seconds) % (60 * 60 * 24)
}

// Optimized version
fn parse_time_optimized(time_str: &str) -> u32 {
    let mut parts = time_str.split(':');
    let hours: u32 = parts.next().unwrap().parse().unwrap();
    let minutes: u32 = parts.next().unwrap().parse().unwrap();
    let seconds: u32 = parts.next().unwrap().parse().unwrap();
    assert!(parts.next().is_none(), "Invalid time format");
    assert!(seconds == 0);
    (hours * 3600 + minutes * 60 + seconds) % (60 * 60 * 24)
}

// Bytes version
fn parse_time_bytes(time_str: &str) -> u32 {
    let bytes = time_str.as_bytes();
    if bytes.len() != 8 {
        panic!("Invalid time format");
    }
    let hours = (bytes[0] - b'0') as u32 * 10 + (bytes[1] - b'0') as u32;
    assert!(bytes[2] == b':');
    let minutes = (bytes[3] - b'0') as u32 * 10 + (bytes[4] - b'0') as u32;
    assert!(bytes[5] == b':');
    let seconds = (bytes[6] - b'0') as u32 * 10 + (bytes[7] - b'0') as u32;
    assert!(seconds == 0);
    (hours * 3600 + minutes * 60 + seconds) % (60 * 60 * 24)
}

// Generate random time string
fn generate_random_time() -> String {
    let mut rng = rand::rng();
    let hours = rng.random_range(0..24);
    let minutes = rng.random_range(0..60);
    format!("{:02}:{:02}:00", hours, minutes)
}

// Generate random time strings with padding
fn generate_random_time_with_padding() -> String {
    let mut rng = rand::rng();
    let hours = rng.random_range(0..24);
    let minutes = rng.random_range(0..60);
    let pad_left = rng.random_range(0..5);
    let pad_right = rng.random_range(0..5);
    format!(
        "{:>width1$}{:02}:{:02}:00{:width2$}",
        "",
        hours,
        minutes,
        "",
        width1 = pad_left,
        width2 = pad_right
    )
}

fn criterion_benchmark(c: &mut Criterion) {
    let mut rng = rand::rng();

    // Generate test data
    const SAMPLE_SIZE: usize = 1000;
    let standard_times: Vec<String> = (0..SAMPLE_SIZE)
        .map(|_| generate_random_time())
        .collect();
    let padded_times: Vec<String> = (0..SAMPLE_SIZE)
        .map(|_| generate_random_time_with_padding())
        .collect();

    let mut group = c.benchmark_group("Time Parser");

    // Configure the benchmark
    group.measurement_time(Duration::from_secs(10));
    group.sample_size(100);

    // Benchmark with standard format
    group.bench_function("original/standard", |b| {
        b.iter(|| {
            let idx = rng.random_range(0..SAMPLE_SIZE);
            parse_time_original(black_box(standard_times[idx].clone()))
        })
    });

    group.bench_function("optimized/standard", |b| {
        b.iter(|| {
            let idx = rng.random_range(0..SAMPLE_SIZE);
            parse_time_optimized(black_box(&standard_times[idx]))
        })
    });

    group.bench_function("bytes/standard", |b| {
        b.iter(|| {
            let idx = rng.random_range(0..SAMPLE_SIZE);
            parse_time_bytes(black_box(&standard_times[idx]))
        })
    });

    // Benchmark with padded format
    group.bench_function("original/padded", |b| {
        b.iter(|| {
            let idx = rng.random_range(0..SAMPLE_SIZE);
            parse_time_original(black_box(padded_times[idx].clone()))
        })
    });

    group.bench_function("optimized/padded", |b| {
        b.iter(|| {
            let idx = rng.random_range(0..SAMPLE_SIZE);
            parse_time_optimized(black_box(&padded_times[idx]))
        })
    });

    group.finish();
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);