const mediaQuery = window.matchMedia("(prefers-color-scheme: light)");
document.documentElement.setAttribute(
    "data-theme",
    mediaQuery.matches ? "l" : "d"
);
mediaQuery.addEventListener("change", (e): void => {
    document.documentElement.setAttribute("data-theme", e.matches ? "l" : "d");
});

const themeSwitcher = document.querySelector<HTMLSelectElement>("#theme");

themeSwitcher?.addEventListener("change", (): void => {
    if (themeSwitcher.value === "l") {
        document.documentElement.setAttribute("data-theme", "l");
    } else if (themeSwitcher.value === "d") {
        document.documentElement.setAttribute("data-theme", "d");
    } else if (themeSwitcher.value === "s") {
        document.documentElement.setAttribute(
            "data-theme",
            mediaQuery.matches ? "l" : "d"
        );
    }
});

function generateHTML(legs) {
    const container = document.createElement("div");

    for (const leg of legs) {
        const legDiv = document.createElement("div");
        legDiv.className = "itinerary-leg-details";

        // Calculate duration
        const fromTime = new Date(`2024-01-01T${leg.from.time}`);
        const toTime = new Date(`2024-01-01T${leg.to.time}`);
        const durationMs = toTime - fromTime;
        const durationMins = Math.floor(durationMs / (1000 * 60));
        const durationSecs = Math.floor((durationMs % (1000 * 60)) / 1000);

        // Format times for display
        const displayFromTime = leg.from.time.substr(0, 5);
        const displayToTime = leg.to.time.substr(0, 5);

        // Create times div
        const timesDiv = document.createElement("div");
        timesDiv.className = "times";
        if (leg.route === "WALK") {
            if (durationMins < 2) {
                continue;
            }
            // For walking segments, show duration in minutes and seconds
            timesDiv.textContent = durationSecs
                ? `${durationMins} mins ${durationSecs} secs`
                : `${durationMins} mins`;
        } else {
            // For transit segments, estimate distance based on duration (rough approximation)
            const estimatedKm = (durationMins / 4).toFixed(1); // Assuming average speed of 15 km/h
            // timesDiv.textContent = `${estimatedKm} km, ${durationMins} mins`;
            timesDiv.textContent = `${durationMins} mins`;
        }

        // Create time spans
        const timeSpan = document.createElement("div");
        timeSpan.innerHTML = `<span>${displayFromTime}</span> - <span>${displayToTime}</span>`;

        // Create mode div
        const modeDiv = document.createElement("div");
        modeDiv.className = "mode";

        if (leg.route === "WALK") {
            modeDiv.innerHTML = `<b>pěšky</b>`;
            if (leg.to.place === "Destination") {
                leg.to.place = "Cíl";
            }
            if (leg.from.place === "Origin") {
                leg.from.place = "Start";
            }
            modeDiv.innerHTML += `<a>${leg.from.place} → ${leg.to.place}</a>`;
        } else {
            modeDiv.innerHTML = `
                <b>bus/trolejbus</b>
                <!-- <a>${leg.route} ${leg.to.place}</a>, -->
                <a>${leg.route}</a>
                <b></b>
                <br /><a>${leg.from.place}</a>
                →
                <a>${leg.to.place}</a>
            `;
        }

        // Assemble the leg div
        legDiv.appendChild(timesDiv);
        legDiv.appendChild(timeSpan);
        legDiv.appendChild(modeDiv);

        container.appendChild(legDiv);
    }

    return container.outerHTML;
}

const prefetchManager = {
    cache: new Map(),
    inFlightRequests: new Map(),

    async prefetch(url, options = {}) {
        console.log(`pre-flight start ${new Date().toISOString()}`);

        // If there's already an in-flight request, return its promise
        if (this.inFlightRequests.has(url.toString())) {
            return this.inFlightRequests.get(url.toString());
        }

        // If it's cached, return cached data
        if (this.cache.has(url.toString())) {
            return this.cache.get(url.toString());
        }

        // Create and store the fetch promise
        const fetchPromise = fetch(url, {
            ...options,
            headers: {
                ...options.headers,
                Purpose: "prefetch",
            },
        })
            .then((response) => response.json())
            .then((data) => {
                // Store in cache once complete
                this.cache.set(url.toString(), data);
                // Remove from in-flight requests
                console.log(`in-flight completed ${new Date().toISOString()}`);
                this.inFlightRequests.delete(url.toString());
                return data;
            })
            .catch((error) => {
                this.inFlightRequests.delete(url.toString());
                throw error;
            });

        // Store the in-flight promise
        this.inFlightRequests.set(url.toString(), fetchPromise);
        return fetchPromise;
    },
};

document.addEventListener("DOMContentLoaded", () => {
    const submitButton = document.querySelector("#tripForm > button");
    const form = document.querySelector("#tripForm");
    let prefetchTimer;
    let currentPrefetchPromise = null;

    submitButton.addEventListener("mouseenter", () => {
        prefetchTimer = setTimeout(() => {
            const url = createUrl();

            // Store the prefetch promise
            currentPrefetchPromise = prefetchManager.prefetch(url);
        }, 10);
    });

    submitButton.addEventListener("mouseleave", () => {
        clearTimeout(prefetchTimer);
    });

    form.addEventListener("submit", async (e) => {
        e.preventDefault();

        const url = createUrl();

        let json;
        try {
            // If we have an in-flight prefetch, wait for it
            if (currentPrefetchPromise) {
                console.log("prefetch in-flight");
                json = await currentPrefetchPromise;
            } else {
                // No prefetch in progress, make a new request
                const response = await fetch(url);
                json = await response.json();
            }
        } catch (error) {
            console.error("Request failed:", error);
        }

        document.getElementById("tripForm")!.style.display = "none";

        let b = "<div style='display: grid; zoom: 1.25;'><br><br>";
        for (const element of json) {
            b += generateHTML(
                element
                    .map((segment) => {
                        return {
                            from: {
                                place: segment.from,
                                time: segment.start_time,
                            },
                            to: {
                                place: segment.to,
                                time: segment.end_time,
                            },
                            route: segment.route,
                        };
                    })
                    .filter((leg) => leg !== null)
            );
            b += "<br><br>";
        }
        b += "</div>";

        document.body.innerHTML = b;

        let history = localStorage.getItem("history");
        if (history == null || history === "") {
            history = [];
        } else {
            history = JSON.parse(history);
        }
        history.push({
            from: url.searchParams.get("from"),
            to: url.searchParams.get("to"),
            favorite: false,
        });
        localStorage.setItem("history", JSON.stringify(history));

        console.log(json);
    });

    let history = JSON.parse(localStorage.getItem("history"));
    for (const element of history) {
        document.getElementById("history").innerHTML += `Odkud: ${
            element.from
        }<br>Kam: ${element.to}<br>Oblíbené:${
            element.favorite ? "Ano" : "Ne"
        }<br><br>`;
    }

    document.getElementById("favorite").addEventListener("click", function () {
        let history = JSON.parse(localStorage.getItem("history"));
        history[history.length - 1].favorite = true;
        localStorage.setItem("history", JSON.stringify(history));

        document.getElementById("history").innerHTML = "";
        for (const element of history) {
            document.getElementById("history").innerHTML += `Odkud: ${
                element.from
            }<br>Kam: ${element.to}<br>Oblíbené:${
                element.favorite ? "Ano" : "Ne"
            }<br><br>`;
        }
    });
});

// Get the elements
const advancedToggle = document.getElementById("advancedToggle");
const advancedFilters = document.getElementById("advancedFilters");
const arrow = document.querySelector(".arrow-down");

// Add click event to toggle advanced filters
advancedToggle.addEventListener("click", function () {
    advancedFilters.classList.toggle("open");
    arrow.classList.toggle("open");
});

function createUrl() {
    const url = new URL("http://127.0.0.1:8081/plan");

    url.searchParams.append(
        "from",
        document.querySelector<HTMLInputElement>("#from")?.dataset.gps ??
            document.querySelector<HTMLInputElement>("#from")?.value ??
            ""
    );
    url.searchParams.append(
        "to",
        document.querySelector<HTMLInputElement>("#to")?.dataset.gps ??
            document.querySelector<HTMLInputElement>("#to").value ??
            ""
    );
    const dateValue = document.querySelector<HTMLInputElement>("#date")?.value;
    if (dateValue) {
        url.searchParams.append("date", dateValue);
    }
    const timeValue = document.querySelector<HTMLInputElement>("#time")?.value;
    if (timeValue) {
        url.searchParams.append("time", timeValue);
    }

    for (const element of [
        "max-transfers",
        "min-time-transfers",
        "max-time-transfers",
        "max-walk-time",
    ]) {
        const searchParam = document.querySelector<HTMLInputElement>(
            `#${element}`
        )?.value;
        if (searchParam) {
            url.searchParams.append(element, searchParam);
        }
    }

    return url;
}
