interface Coordinates {
    lat: number;
    lng: number;
}

const coordinatesInputFrom = document.getElementById(
    "from"
) as HTMLInputElement;
const coordinatesInputTo = document.getElementById("to") as HTMLInputElement;
const mapToggleBtn = document.querySelectorAll("#mapToggleFrom, #mapToggleTo");
const mapModal = document.getElementById("mapModal") as HTMLDivElement;
const confirmBtn = document.getElementById("confirmBtn") as HTMLButtonElement;
const cancelBtn = document.getElementById("cancelBtn") as HTMLButtonElement;
const locationForm = document.getElementById("locationForm") as HTMLFormElement;

let map: L.Map | null = null;
let marker: L.Marker | null = null;
let selectedPosition: Coordinates = {
    lat: 49.8,
    lng: 15.4,
};

// Initialize the map (only when needed)
function initializeMap(): void {
    // If map already exists, just update the view
    if (map) {
        map.setView([selectedPosition.lat, selectedPosition.lng], 13);
        marker?.setLatLng([selectedPosition.lat, selectedPosition.lng]);
        return;
    }

    // Create new map instance
    const mapElement = document.getElementById("map") as HTMLElement;
    map = L.map(mapElement).setView(
        [selectedPosition.lat, selectedPosition.lng],
        7
    );

    // Add tile layer (OpenStreetMap)
    L.tileLayer("https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png", {
        attribution:
            '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
    }).addTo(map);

    // Add marker at current position
    marker = L.marker([selectedPosition.lat, selectedPosition.lng], {
        draggable: true,
    }).addTo(map);

    // Update position when marker is dragged
    marker.on("dragend", () => {
        if (marker) {
            const position = marker.getLatLng();
            selectedPosition = { lat: position.lat, lng: position.lng };
        }
    });

    // Update position when map is clicked
    map.on("click", (e: L.LeafletMouseEvent) => {
        selectedPosition = { lat: e.latlng.lat, lng: e.latlng.lng };
        marker?.setLatLng(e.latlng);
    });
}

function parseCoordinates(input: string): Coordinates | null {
    const parts = input.split(",").map((part) => part.trim());

    if (parts[0] === undefined || parts[1] === undefined) return null;

    const lat = Number.parseFloat(parts[0]);
    const lng = Number.parseFloat(parts[1]);

    if (Number.isNaN(lat) || Number.isNaN(lng)) return null;

    return { lat, lng };
}

function formatCoordinates(coords: Coordinates): string {
    return `${coords.lat.toFixed(8)}, ${coords.lng.toFixed(8)}`;
}

let lastMapToggle = "";

// Event listeners
for (const element of mapToggleBtn) {
    element.addEventListener("click", (e: Event) => {
        if (e.target?.id) {
            lastMapToggle = e.target.id;
        }

        mapModal.style.display = "block";

        // Try to parse coordinates from input
        const inputCoords = parseCoordinates(
            lastMapToggle === "mapToggleFrom"
                ? coordinatesInputFrom.value
                : coordinatesInputTo.value
        );
        if (inputCoords) {
            selectedPosition = inputCoords;
        }

        initializeMap();
        // Force map to recalculate its size
        map?.invalidateSize();
    });
}

confirmBtn.addEventListener("click", () => {
    // Update input with selected coordinates
    if (lastMapToggle === "mapToggleFrom") {
        coordinatesInputFrom.value = formatCoordinates(selectedPosition);
    } else {
        coordinatesInputTo.value = formatCoordinates(selectedPosition);
    }
    mapModal.style.display = "none";
});

cancelBtn.addEventListener("click", () => {
    // Close modal without saving changes
    mapModal.style.display = "none";
});

// Close modal if clicked outside the map container
mapModal.addEventListener("click", (event) => {
    if (event.target === mapModal) {
        mapModal.style.display = "none";
    }
});
