use bincode::config;
use bincode::Decode;
use bincode::Encode;
use cached::proc_macro::cached;
use core::f64;
use csv::{QuoteStyle, WriterBuilder};
use geoutils::{Distance, Location};
use log::{debug, info, warn};
use rayon::iter::IntoParallelIterator;
use rayon::iter::ParallelIterator;
use regex::Regex;
use reqwest::blocking::Client;
use reqwest::header;
use reqwest::header::HeaderMap;
use rocksdb::{Options, DB};
use serde::Deserialize;
use serde::Serialize;
use serde_json::Value;
use std::collections::HashMap;
use std::collections::HashSet;
use std::env;
use std::fs;
use std::hash::Hash;
use std::hash::Hasher;
use std::io::Write;
use std::panic;
use std::panic::AssertUnwindSafe;
use std::path::Path;
use std::path::PathBuf;
use std::process;
use std::sync::Arc;
use std::sync::Mutex;
use std::sync::OnceLock;
use std::thread;
use strsim::levenshtein;
use walkdir::WalkDir;

#[derive(Debug, Deserialize, Serialize)]
struct Stop {
    stop_id: String,
    stop_code: String,
    stop_name: String,
    stop_desc: String,
    stop_lat: String,
    stop_lon: String,
    zone_id: String,
    stop_url: String,
    location_type: String,
    parent_station: String,
    stop_timezone: String,
    wheelchair_boarding: String,
    platform_code: String,
}

#[derive(Debug, Deserialize, Clone)]
#[allow(dead_code)]
struct StopTimes {
    trip_id: String,
    arrival_time: String,
    departure_time: String,
    stop_id: String,
    stop_sequence: String,
    stop_headsign: String,
    pickup_type: String,
    drop_off_type: String,
    shape_dist_traveled: i32,
    timepoint: String,
    stop_zone_ids: String,
}

const LIMIT: usize = 100;
const LIMIT_MEILISEARCH: usize = 200;
const NOMINATIM_URL: &str = "http://127.0.0.1:8080";

struct CleanupOnDrop<F>
where
    F: FnMut(),
{
    callback: F,
    only_on_panic: bool,
}

impl<F> Drop for CleanupOnDrop<F>
where
    F: FnMut(),
{
    fn drop(&mut self) {
        if !self.only_on_panic || thread::panicking() {
            (self.callback)();
        }
    }
}

fn main() {
    env_logger::init();

    let _cleanup = CleanupOnDrop {
        callback: || println!("failed panic error"),
        only_on_panic: true,
    };

    let args: Vec<String> = env::args().collect();
    if args.len() != 2 {
        eprintln!("Usage: {} <folder_path>", args[0]);
        process::exit(1);
    }
    let folder_path = &args[1];
    if !Path::new(folder_path).is_dir() {
        eprintln!("Error: Folder not found: {folder_path}");
        process::exit(1);
    }
    let folders: Vec<PathBuf> = WalkDir::new(folder_path)
        .max_depth(1)
        .min_depth(1)
        .into_iter()
        .filter_map(std::result::Result::ok)
        .filter(|e| e.file_type().is_dir())
        .map(|e| e.path().to_owned())
        .collect();

    let number_of_errors = Arc::new(Mutex::new(0));

    folders.into_iter().for_each(|i| {
        process_folder(&i, &number_of_errors);
    });

    info!("Completed");
    warn!("Number of errors: {}", *number_of_errors.lock().unwrap());
}

fn client() -> &'static Client {
    static CLIENT: OnceLock<Client> = OnceLock::new();
    CLIENT.get_or_init(Client::new)
}

#[cached]
fn get_locations(stop_name: String, use_nominatim: bool) -> (Vec<(Location, String)>, bool) {
    let mut locations: Vec<(Location, String)> = Vec::new();

    if use_nominatim {
        let url = NOMINATIM_URL.to_owned() + "/search";

        let mut okres = "";
        if stop_name.contains('[') {
            okres = match get_bracket_content(stop_name.as_str()).as_str() {
                // https://cs.wikipedia.org/wiki/Seznam_okres%C5%AF_v_%C4%8Cesku
                // https://csu.gov.cz/produkty-archiv/13-2199-04-2004-regions_and_districts___abbreviations
                // https://cs.wikipedia.org/wiki/Okres_Brno-venkov
                "BN" => "okres Benešov",
                "BE" => "okres Beroun",
                "BK" => "okres Blansko",
                "BM" => "okres Brno-město",
                "BI" | "BO" => "okres Brno-venkov",
                "BR" => "okres Bruntál",
                "BV" => "okres Břeclav",
                "CL" => "okres Česká Lípa",
                "CB" => "okres České Budějovice",
                "CK" => "okres Český Krumlov",
                "DC" => "okres Děčín",
                "DO" => "okres Domažlice",
                "FM" => "okres Frýdek-Místek",
                "HB" => "okres Havlíčkův Brod",
                "HO" => "okres Hodonín",
                "HK" => "okres Hradec Králové",
                "CH" => "okres Cheb",
                "CV" => "okres Chomutov",
                "CR" => "okres Chrudim",
                "JN" => "okres Jablonec nad Nisou",
                "JE" => "okres Jeseník",
                "JC" => "okres Jičín",
                "JI" => "okres Jihlava",
                "JH" => "okres Jindřichův Hradec",
                "KV" => "okres Karlovy Vary",
                "KI" => "okres Karviná",
                "KD" => "okres Kladno",
                "KT" => "okres Klatovy",
                "KO" => "okres Kolín",
                "KM" => "okres Kroměříž",
                "KH" => "okres Kutná Hora",
                "LI" => "okres Liberec",
                "LT" => "okres Litoměřice",
                "LN" => "okres Louny",
                "ME" => "okres Mělník",
                "MB" => "okres Mladá Boleslav",
                "MO" => "okres Most",
                "NA" => "okres Náchod",
                "NJ" => "okres Nový Jičín",
                "NB" => "okres Nymburk",
                "OC" => "okres Olomouc",
                "OP" => "okres Opava",
                "OV" => "okres Ostrava-město",
                "PU" => "okres Pardubice",
                "PE" => "okres Pelhřimov",
                "PI" => "okres Písek",
                "PJ" => "okres Plzeň-jih",
                "PM" => "okres Plzeň-město",
                "PS" => "okres Plzeň-sever",
                "PY" => "okres Praha-východ",
                "PZ" => "okres Praha-západ",
                "PT" => "okres Prachatice",
                "PV" => "okres Prostějov",
                "PR" => "okres Přerov",
                "PB" => "okres Příbram",
                "RA" => "okres Rakovník",
                "RO" => "okres Rokycany",
                "RK" => "okres Rychnov nad Kněžnou",
                "SM" => "okres Semily",
                "SO" => "okres Sokolov",
                "ST" => "okres Strakonice",
                "SY" => "okres Svitavy",
                "SU" => "okres Šumperk",
                "TA" => "okres Tábor",
                "TC" => "okres Tachov",
                "TP" => "okres Teplice",
                "TU" => "okres Trutnov",
                "TR" => "okres Třebíč",
                "UH" => "okres Uherské Hradiště",
                "UL" => "okres Ústí nad Labem",
                "UO" => "okres Ústí nad Orlicí",
                "VS" => "okres Vsetín",
                "VY" => "okres Vyškov",
                "ZL" => "okres Zlín",
                "ZN" => "okres Znojmo",
                "ZR" => "okres Žďár nad Sázavou",
                "AB" => "",
                _ => unreachable!("{}", dbg!(stop_name)),
            }
        }

        let stop_name_edited = if okres.is_empty() {
            stop_name.replace(',', " ")
        } else {
            remove_brackets(&stop_name.replace(',', " "))
        };

        let query = [
            ("q".to_owned(), stop_name_edited + " Bus stop"),
            ("dedupe".to_owned(), "0".to_owned()),
            ("limit".to_owned(), LIMIT.to_string()),
            ("namedetails".to_owned(), "1".to_owned()),
        ];

        let response = make_get_request(&url, &query);

        debug!("Status: {}", response.status);
        assert!((200..300).contains(&response.status));
        debug!("Headers:\n{:#?}", response.headers);

        let body = response.body;
        debug!("Body:\n{body}");

        let v: Value = serde_json::from_str(&body).unwrap();

        assert!(v.as_array().unwrap().len() != LIMIT, "{v:#?} {stop_name}");

        for x in v.as_array().unwrap() {
            debug!("{}", x["lat"]);
            debug!("{}", x["lon"]);
            if !okres.is_empty() && !x["display_name"].as_str().unwrap().contains(okres) {
                continue;
            }
            locations.push((
                Location::new(x["lat"].as_str().unwrap().parse::<f64>().unwrap(), x["lon"].as_str().unwrap().parse::<f64>().unwrap()),
                // x["name"].as_str().unwrap().to_owned(),
                x["namedetails"]["official_name"].as_str().unwrap_or_else(|| x["name"].as_str().unwrap()).to_owned(),
            ));
        }
    }

    if locations.is_empty() {
        let mut headers = header::HeaderMap::new();
        headers.insert("Authorization", "Bearer abcd".parse().unwrap());
        headers.insert("Content-Type", "application/json".parse().unwrap());

        let res = make_post_request(
            "http://127.0.0.1:7700/indexes/bus_tram_stops/search",
            headers,
            format!(
                r#"
{{

	"q": "{stop_name}",
	"attributesToRetrieve": [
		"name",
		"_geo"
	],
	"offset": 0,
	"limit": {LIMIT_MEILISEARCH},
	"showRankingScore": false,
	"showRankingScoreDetails": false
}}

"#
            ),
        );
        debug!("stop_name: {stop_name}");
        debug!("{:#?}", res);

        let v: Value = serde_json::from_str(&res.body).unwrap();

        debug!("{v:#?}");

        // assert!(v.as_array().unwrap().len() != LIMIT_MEILISEARCH);
        // assert!(v["hits"].as_array().unwrap().len() != LIMIT_MEILISEARCH);
        for x in v["hits"].as_array().unwrap() {
            // debug!("{}", x["lat"]);
            // debug!("{}", x["lon"]);
            locations.push((
                Location::new(
                    x["_geo"].as_str().unwrap().split(',').next().unwrap().parse::<f64>().unwrap(),
                    x["_geo"].as_str().unwrap().split(',').last().unwrap().parse::<f64>().unwrap(),
                ),
                x["name"].as_str().unwrap().to_owned(),
            ));
        }
        debug!("{locations:#?}");
        return (locations, false);
    }

    return (locations, true);
}

fn remove_brackets(s: &str) -> String {
    let mut result = String::with_capacity(s.len());
    let mut skip = false;
    let mut open_bracket_count = 0;
    let mut close_bracket_count = 0;

    for c in s.chars() {
        match c {
            '[' => {
                skip = true;
                if open_bracket_count != 0 {
                    unreachable!()
                }
                open_bracket_count += 1;
            }
            ']' => {
                skip = false;
                if close_bracket_count != 0 {
                    unreachable!()
                }
                close_bracket_count += 1;
            }
            _ if !skip => result.push(c),
            _ => {}
        }
    }

    result
}

fn get_bracket_content(s: &str) -> String {
    if let (Some(start), Some(end)) = (s.find('['), s.find(']')) {
        if start < end {
            s[start + 1..end].to_string()
        } else {
            unreachable!()
        }
    } else {
        unreachable!()
    }
}

#[derive(Clone)]
struct HashableLocation(geoutils::Location);

impl HashableLocation {}

impl Hash for HashableLocation {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.0.latitude().to_bits().hash(state);
        self.0.longitude().to_bits().hash(state);
    }
}

impl PartialEq for HashableLocation {
    fn eq(&self, other: &Self) -> bool {
        self.0.latitude() == other.0.latitude() && self.0.longitude() == other.0.longitude()
    }
}

impl Eq for HashableLocation {}

// #[cached]
fn are_stops_in_cadastral_place(stops: Vec<(HashableLocation, String)>, stop_name: String) -> bool {
    let parts: Vec<&str> = stop_name.split(',').collect();
    assert!(parts.len() <= 3);
    if parts.len() != 3 || parts[1].trim().is_empty() {
        return true;
    }
    for (location, name) in stops {
        let url = format!("{}/reverse?lat={}&lon={}&zoom=14&format=json", NOMINATIM_URL, location.0.latitude(), location.0.longitude());

        let response = make_get_request(&url, &[]);

        assert!((200..300).contains(&response.status), "{response:#?}");

        let v: Value = serde_json::from_str(&response.body).unwrap();
        debug!("{:#?}", v);

        match v["name"].as_str() {
            Some(v) => {
                if v.trim().to_lowercase().contains(&parts[1].trim().to_lowercase()) {
                    continue;
                }
                let parts_: Vec<&str> = name.split(',').collect();
                assert!(parts_.len() <= 3);
                if parts_.len() != 3 || parts_[1].trim().is_empty() {
                    return false;
                }
                if parts[1].trim().to_lowercase() == parts_[1].trim().to_lowercase() {
                    continue;
                }
                return false;
            }
            _ => {
                panic!();
            }
        }
    }

    true
}

fn convert_to_hashable(original: Vec<(geoutils::Location, String)>) -> Vec<(HashableLocation, String)> {
    original.into_iter().map(|(location, string)| (HashableLocation(location), string)).collect()
}

fn if_other_stop_have_same_location(stop_points: &mut Vec<(String, Location)>, current_index: usize, center: Location, location_name: &str) -> bool {

    for i in 0..stop_points.len() {
        if !(i != current_index && stop_points[i].1.latitude() == center.latitude() && stop_points[i].1.longitude() == center.longitude() && stop_points[i].0 != stop_points[current_index].0) {
            continue;
        }

        let mut other_stop_distance = levenshtein(location_name, &stop_points[i].0);
        let current_stop_distance = if location_name
            == *stop_points[current_index]
                .0
                .split(',')
                .collect::<Vec<&str>>()
                .get(stop_points[current_index].0.chars().filter(|c| *c == ',').count())
                .unwrap_or(&"")
        {
            0
        } else {
            levenshtein(location_name, &stop_points[current_index].0)
        };

        if location_name
            == *stop_points[i]
                .0
                .split(',')
                .collect::<Vec<&str>>()
                .get(stop_points[i].0.chars().filter(|c| *c == ',').count())
                .unwrap_or(&"")
        {
            other_stop_distance = 0;
        }

        if other_stop_distance == current_stop_distance {
            debug!("{:#?}", stop_points);
            if current_stop_distance > 5 || (current_stop_distance == 5 && location_name.chars().count() < 15) {
                stop_points[i].1 = Location::new(0, 0);
                return true;
            }
            panic!("stop_points: {stop_points:#?}, current_index: {current_index}, center: {center:#?}, location_name: {location_name:#?}, other_stop_distance: {other_stop_distance}, current_stop_distance: {current_stop_distance}, location_name len: {}", location_name.chars().count());
        } else if other_stop_distance < current_stop_distance {
            return true;
        } else {
            stop_points[i].1 = Location::new(0, 0);
            return false;
        }
    }
    false
}

#[derive(Clone, Serialize, Deserialize, Debug, Decode, Encode)]
struct ResponseCustom {
    body: String,
    status: u16,                    // Store status code as u16
    headers: Vec<(String, String)>, // HeaderMap as Vec for serialization
}

#[derive(Hash, Eq, PartialEq, Serialize, Encode)]
struct GetRequestKey {
    url: String,
    query: Vec<(String, String)>,
}

static DB_INSTANCE: OnceLock<DB> = OnceLock::new();

fn db() -> &'static DB {
    DB_INSTANCE.get_or_init(|| {
        let mut opts = Options::default();
        opts.create_if_missing(true);
        opts.increase_parallelism(12);
        opts.set_allow_mmap_reads(true);
        opts.set_allow_mmap_writes(true);

        DB::open(&opts, "cache.db").expect("Failed to open RocksDB")
    })
}

fn make_get_request(url: &str, query: &[(String, String)]) -> ResponseCustom {
    let config = config::standard();

    // Create cache key and serialize it
    let key = GetRequestKey {
        url: url.to_owned(),
        query: query.to_owned(),
    };
    let key_bytes = bincode::encode_to_vec(&key, config).expect("Failed to serialize key");

    // Try to get from cache
    if let Ok(Some(cached_value)) = db().get(&key_bytes) {
        if let Ok(cached_response) = bincode::decode_from_slice(&cached_value, config) {
            return cached_response.0;
        }
        warn!("failed to deserialize cached_value");
    }

    // If not in cache, make the request
    let response = client().get(url).query(&query).send().unwrap();

    // Convert headers to Vec<(String, String)>
    let headers = response.headers().iter().map(|(k, v)| (k.as_str().to_string(), v.to_str().unwrap_or("").to_string())).collect();

    let response_custom = ResponseCustom {
        status: response.status().as_u16(),
        body: response.text().unwrap(),
        headers,
    };

    // Cache the response
    let value_bytes = bincode::encode_to_vec(&response_custom, config).expect("Failed to serialize response");
    db().put(key_bytes, value_bytes).expect("Failed to write to cache");

    response_custom
}

#[derive(Hash, Eq, PartialEq, Serialize, Encode)]
struct PostRequestKey {
    url: String,
    headers: Vec<(String, String)>,
    body: String,
}

fn make_post_request(url: &str, headers: HeaderMap, body: String) -> ResponseCustom {
    let config = config::standard();

    // Convert headers to Vec<(String, String)>
    let headers_vec = headers.iter().map(|(k, v)| (k.as_str().to_string(), v.to_str().unwrap_or("").to_string())).collect();

    // Create cache key and serialize it
    let key = PostRequestKey {
        url: url.to_owned(),
        headers: headers_vec,
        body: body.clone(),
    };
    let key_bytes = bincode::encode_to_vec(&key, config).expect("Failed to serialize key");

    // Try to get from cache
    if let Ok(Some(cached_value)) = db().get(&key_bytes) {
        if let Ok(cached_response) = bincode::decode_from_slice(&cached_value, config) {
            return cached_response.0;
        }
        warn!("failed to deserialize cached_value");
    }

    // If not in cache, make the request
    let response = client().post(url).headers(headers).body(body).send().unwrap();

    // Convert headers to Vec<(String, String)>
    let headers = response.headers().iter().map(|(k, v)| (k.as_str().to_string(), v.to_str().unwrap_or("").to_string())).collect();

    let response_custom = ResponseCustom {
        status: response.status().as_u16(),
        body: response.text().unwrap(),
        headers,
    };

    // Cache the response
    let value_bytes = bincode::encode_to_vec(&response_custom, config).expect("Failed to serialize response");
    db().put(key_bytes, value_bytes).expect("Failed to write to cache");

    response_custom
}

fn group_locations(mut locations: Vec<(Location, String)>, distance: Distance) -> Vec<(Location, Vec<(Location, String)>)> {
    // Group locations that are within 300m of each other
    let mut locations_groups = Vec::new();
    while !locations.is_empty() {
        let first_location = locations[0].clone();

        // Partition locations into those within and outside 300m of first location
        let (group_, remainder): (Vec<(Location, String)>, Vec<(Location, String)>) = locations
            .into_iter()
            .partition(|x| x.0.is_in_circle(&first_location.0, distance).unwrap() && stop_names_distance(&x.1, &first_location.1) == 0);
        let mut locations_refs: Vec<&Location> = group_.iter().map(|x| &x.0).collect();
        let center = Location::center(&locations_refs);
        let (group, remainder): (Vec<(Location, String)>, Vec<(Location, String)>) = remainder
            .into_iter()
            .partition(|x| x.0.is_in_circle(&center, distance).unwrap() && stop_names_distance(&x.1, &first_location.1) == 0);
        locations_refs.extend(group.iter().map(|x| &x.0).collect::<Vec<_>>());

        locations = remainder;

        let mut groups_merged = group_.clone();
        groups_merged.extend(group.clone());
        if groups_merged.is_empty() {
            println!();
        }
        assert!(!groups_merged.is_empty(), "{locations:#?}");

        // Calculate center of the group
        locations_groups.push((Location::center(&locations_refs), groups_merged));
    }

    locations_groups
}
fn group_locations_(mut locations: Vec<(Location, String)>, distance: Distance) -> Vec<(Location, Vec<(Location, String)>)> {
    // Group locations that are within 300m of each other
    assert!(distance != Distance::from_meters(0));
    let mut locations_groups = Vec::new();
    while !locations.is_empty() {
        let first_location = locations[0].clone();

        // Partition locations into those within and outside 300m of first location
        let (mut group, mut remainder): (Vec<(Location, String)>, Vec<(Location, String)>) = locations.into_iter().partition(|x| x.0.is_in_circle(&first_location.0, distance).unwrap());
        loop {
            let (group_, remainder_): (Vec<(Location, String)>, Vec<(Location, String)>) = remainder.clone().into_iter().partition(|x| group.iter().any(|y| y.0.is_in_circle(&x.0, distance).unwrap()));
            if group_.is_empty() {
                break;
            }
            group.extend(group_);
            remainder = remainder_;
        }

        locations = remainder;

        let locations_refs: Vec<&Location> = group.iter().map(|x| &x.0).collect();
        // Calculate center of the group
        locations_groups.push((Location::center(&locations_refs), group));
    }

    locations_groups
}

fn find_min_distance_index(locations_groups: &[(Location, Vec<(Location, String)>)], center: Location) -> usize {
    let mut min_distance = f64::MAX;
    let mut min_distance_index = 0;

    for (iiii, item) in locations_groups.iter().enumerate() {
        let distance = item.0.distance_to(&center).unwrap().meters();

        if distance < min_distance {
            min_distance = distance;
            min_distance_index = iiii;
        }
    }

    min_distance_index
}

fn process_folder(i: &PathBuf, number_of_errors: &Arc<Mutex<usize>>) {
    let mut rdr = csv::Reader::from_path(format!("{}/stop_times.txt", i.to_str().unwrap())).unwrap();

    let mut stop_times: HashMap<String, Vec<StopTimes>> = HashMap::new();
    for ii in rdr.deserialize() {
        let record: StopTimes = ii.unwrap();
        stop_times.entry(record.trip_id.to_string()).or_default().push(record);
    }
    debug!("{:#?}", stop_times.iter().map(|x| x.0).collect::<Vec<_>>());

    let stop_times_: HashMap<String, Vec<Vec<StopTimes>>> =
        stop_times
            .clone()
            .into_iter()
            .fold(HashMap::new(), |mut acc: HashMap<String, Vec<Vec<StopTimes>>>, (_key, stop_time_list): (String, Vec<StopTimes>)| {
                acc.entry(stop_time_list.iter().map(|x| x.stop_id.clone() + &x.shape_dist_traveled.to_string()).collect())
                    .or_default()
                    .push(stop_time_list);
                acc
            });
    let stop_times__: HashMap<String, Vec<Vec<StopTimes>>> = stop_times.into_iter().fold(HashMap::new(), |mut acc, (_key, stop_time_list)| {
        let mut previous_distance = 0;
        acc.entry(
            stop_time_list
                .iter()
                .map(|x| {
                    let a = x.stop_id.clone() + &(x.shape_dist_traveled - previous_distance).to_string();
                    previous_distance = x.shape_dist_traveled;
                    a
                })
                .collect(),
        )
        .or_default()
        .push(stop_time_list);
        acc
    });
    assert!(stop_times_.len() == stop_times__.len());
    debug!("abcdefgh\n{:#?}\n{:#?}\nb", i, stop_times_);

    let mut stops_names_forbidden_for_exact_match: Vec<String> = Vec::new();
    let mut max_stop_points_distance = 0;
    let mut does_stop_points_distance_exists = false;
    'outer_outer_outer: loop {
        let mut rdr = csv::Reader::from_path(format!("{}/stops.txt", i.to_str().unwrap())).unwrap();
        let mut stops_records: Vec<Stop> = rdr.deserialize().collect::<Result<_, _>>().unwrap();
        let mut stops_records: Vec<(&mut Stop, Vec<Location>)> = stops_records.iter_mut().map(|x| (x, Vec::new())).collect();
        println!("||||");
        for ii in &stop_times_ {
            // println!("---------------- {} {:?}", ii.0, ii.1);
            println!("---------------- {}", ii.0);
            let mut stop_points: Vec<(String, Location)> = Vec::new();
            let mut stop_points_distances: Vec<i32> = Vec::new();

            let mut last_stop_point_distance = 0;
            let mut last_stop_point_arrival_time = 0.0;
            let mut last_stop_point_stop_id = String::new();

            let mut errors_in_distances = Vec::new();

            let mut first_iteration = true;
            // populate stop_points with stop names with location 0, 0
            for iii in &ii.1[0] {
                if first_iteration {
                    first_iteration = false;
                    assert!(!(iii.shape_dist_traveled != 0 && iii.stop_sequence == "0"));
                } else {
                    let distance = (iii.shape_dist_traveled - last_stop_point_distance) * 1000;
                    stop_points_distances.push(distance);
                    does_stop_points_distance_exists = true;
                    if distance > max_stop_points_distance {
                        max_stop_points_distance = distance;
                    }
                    assert!(distance >= 0);

                    let arrival_time = parse_time(&iii.arrival_time);

                    assert!(arrival_time >= last_stop_point_arrival_time, "{:?} {} {}", iii, arrival_time, last_stop_point_arrival_time);
                    let grjei = arrival_time - last_stop_point_arrival_time;
                    let speed = (distance as f64 / 1000.0) / (grjei as f64 / 3600.0);
                    if speed > 150.0 {
                        if distance == 1000 && grjei == 0.0 {
                        } else {
                            errors_in_distances.push((
                                stops_records.iter().find(|x| x.0.stop_id == last_stop_point_stop_id).unwrap().0.stop_name.clone(),
                                stops_records.iter().find(|x| x.0.stop_id == iii.stop_id).unwrap().0.stop_name.clone(),
                            ));
                        }
                    }
                }

                last_stop_point_distance = iii.shape_dist_traveled;
                last_stop_point_arrival_time = parse_time(&iii.departure_time);
                last_stop_point_stop_id = iii.stop_id.clone();

                stop_points.push((iii.stop_id.clone(), Location::new(0, 0)));
            }

            debug!("stop_points: {stop_points:#?}");

            for (record, _) in &stops_records {
                for stop_point in &mut stop_points {
                    if stop_point.0 == record.stop_id {
                        stop_point.0.clone_from(&(record.stop_name));
                    }
                }
            }

            debug!("stop_points: {stop_points:#?}");
            debug!("stop_points_distances: {stop_points_distances:#?}");
            assert!(stop_points.len() == stop_points_distances.len() + 1);

            for i in &stop_points {
                assert!(i.1.latitude() == 0.0);
                assert!(i.1.longitude() == 0.0);
            }
            let mut stop_points_name_openstreetmap: Vec<String> = vec![String::new(); stop_points.len()];

            let mut vec = (0..stop_points.len()).collect::<Vec<_>>();
            let n = vec.len();
            'outer: for yy in 0..n {
                let mut first_iteration = true;
                let mut last_ii_was_max = false;
                for &iij in &vec {
                    let stop_point_index_for_comparison = if last_ii_was_max || first_iteration { iij + 1 } else { iij - 1 };
                    let stop_point_index_for_comparison_ = if last_ii_was_max || first_iteration { iij } else { iij - 1 };

                    let mut locations: Vec<(Location, String)>;
                    let mut use_nominatim = true;
                    loop {
                        let (locations_, used_nominatim) = get_locations(stop_points[iij].0.clone(), use_nominatim);
                        locations = locations_;

                        if !first_iteration {
                            let mut y = 0;
                            while y < locations.len() {
                                let distance: f64 = locations[y].0.distance_to(&stop_points[stop_point_index_for_comparison].1).unwrap().meters();
                                if distance - f64::from(stop_points_distances[stop_point_index_for_comparison_]) > 1500.0
                            || distance.mul_add(-2.4, f64::from(stop_points_distances[stop_point_index_for_comparison_]))
                                > 1250.0
                                {
                                    locations.remove(y);
                                } else {
                                    y += 1;
                                }
                            }
                        }

                        if !stops_names_forbidden_for_exact_match.contains(&stop_points[iij].0)
                            && locations
                                .iter()
                                .any(|x| x.1.replace(". ", ".").replace(", ", ",").replace(",,", ",") == stop_points[iij].0.replace(". ", ".").replace(", ", ",").replace(",,", ","))
                        {
                            locations.retain(|x| x.1.replace(". ", ".").replace(", ", ",").replace(",,", ",") == stop_points[iij].0.replace(". ", ".").replace(", ", ",").replace(",,", ","));
                        }
                        locations.retain(|x| !stops_names_forbidden_for_exact_match.contains(&x.1));

                        if locations.is_empty() {
                            if used_nominatim {
                                use_nominatim = false;
                                continue;
                            }

                            if yy < n - 1 {
                                let first = vec.remove(0);
                                vec.insert(n - yy - 1, first);
                            }
                            if first_iteration {
                                debug!("{yy} try failed");
                            } else {
                                *number_of_errors.lock().unwrap() += 1;
                            }
                            continue 'outer;
                        }
                        break;
                    }
                    if locations.is_empty() {
                        panic!("");
                    }

                    if !locations.iter().all(|x| x.0.is_in_circle(&locations[0].0, Distance::from_meters(300)).unwrap()) {
                        if yy < n - 1 {
                            let first = vec.remove(0);
                            vec.insert(n - yy - 1, first);
                        }
                        if first_iteration {
                            debug!("{yy} try failed");
                        } else {
                            *number_of_errors.lock().unwrap() += 1;
                        }
                        continue 'outer;
                    }

                    let center = Location::center(&locations.iter().map(|x| &x.0).collect::<Vec<&Location>>());

                    assert!(!center.latitude().is_nan(), "{locations:#?} {center:#?}");
                    assert!(!center.longitude().is_nan());
                    assert!(center.latitude() != 0.0);
                    assert!(center.longitude() != 0.0);

                    if !if_other_stop_have_same_location(&mut stop_points, iij, center, &locations[0].1) {
                        // println!("{:#?}", locations);
                        stop_points[iij].1 = center;
                        if locations[0].1.is_empty() {
                            println!();
                            panic!();
                        }
                        stop_points_name_openstreetmap[iij].clone_from(&locations[0].1);
                    }

                    debug!("{}: {:?}", stop_points[iij].0.as_str(), center);

                    if iij == n - 1 {
                        last_ii_was_max = true;
                    }

                    first_iteration = false;
                }
                break;
            }

            // assign location to stop based on closest stop to center of previous and next stop*
            for iii in 0..stop_points.len() {
                if iii == 0 || iii == stop_points.len() - 1 || stop_points[iii].1.latitude() != 0.0 || stop_points[iii].1.longitude() != 0.0 {
                    continue;
                }

                if stop_points[iii - 1].1.latitude() == 0.0 || stop_points[iii - 1].1.longitude() == 0.0 || stop_points[iii + 1].1.latitude() == 0.0 || stop_points[iii + 1].1.longitude() == 0.0 {
                    continue;
                }

                let (mut locations, mut used_nominatim) = get_locations(stop_points[iii].0.clone(), true);

                if !first_iteration {
                    let mut y = 0;
                    while y < locations.len() {
                        let distance: f64 = locations[y].0.distance_to(&stop_points[iii - 1].1).unwrap().meters();
                        if distance - f64::from(stop_points_distances[iii - 1]) > 1500.0 || distance.mul_add(-2.4, f64::from(stop_points_distances[iii - 1])) > 1250.0 {
                            locations.remove(y);
                        } else {
                            y += 1;
                        }
                    }
                }

                if !stops_names_forbidden_for_exact_match.contains(&stop_points[iii].0)
                    && locations
                        .iter()
                        .any(|x| x.1.replace(", ", ",").replace(",,", ",") == stop_points[iii].0.replace(", ", ",").replace(",,", ","))
                {
                    locations.retain(|x| x.1.replace(", ", ",").replace(",,", ",") == stop_points[iii].0.replace(", ", ",").replace(",,", ","));
                }
                locations.retain(|x| !stops_names_forbidden_for_exact_match.contains(&x.1));

                if locations.is_empty() {
                    continue;
                }

                // Group locations that are within 300m of each other
                let locations_groups = group_locations(locations, Distance::from_meters(300));

                if locations_groups.len() <= 1 {
                    continue;
                }

                if stop_points[iii].0.contains("Vranovice-Kelčice,Vranovice") {
                    println!();
                }

                let min_stop_name_difference_index = find_min_stop_name_difference_index(&locations_groups, &stop_points[iii].0);
                if stop_names_distance(&stop_points[iii].0, &locations_groups[min_stop_name_difference_index].1[0].1) > 0 {
                    continue;
                }

                let center_of_last_and_next_stop = Location::center(&[&stop_points[iii - 1].1, &stop_points[iii + 1].1]);
                let distance_of_last_next_stop = stop_points[iii - 1].1.distance_to(&stop_points[iii + 1].1).unwrap().meters();

                let distance_previous_stop: f64 = locations_groups[min_stop_name_difference_index].0.distance_to(&stop_points[iii - 1].1).unwrap().meters();
                let distance_next_stop: f64 = locations_groups[min_stop_name_difference_index].0.distance_to(&stop_points[iii + 1].1).unwrap().meters();

                let a1 = distance_previous_stop - f64::from(stop_points_distances[iii - 1]) < 1500.0;
                let a2 = distance_previous_stop.mul_add(-2.4, f64::from(stop_points_distances[iii - 1])) < 1750.0;
                let a3 = distance_next_stop - f64::from(stop_points_distances[iii]) < 1500.0;
                let a4 = distance_next_stop.mul_add(-2.4, f64::from(stop_points_distances[iii])) < 1750.0;

                if center_of_last_and_next_stop.distance_to(&locations_groups[min_stop_name_difference_index].0).unwrap().meters() < distance_of_last_next_stop * 0.5
                    || (distance_previous_stop - f64::from(stop_points_distances[iii - 1]) < 1500.0
                        && distance_previous_stop.mul_add(-2.4, f64::from(stop_points_distances[iii - 1])) < 1750.0
                        && distance_next_stop - f64::from(stop_points_distances[iii]) < 1500.0
                        && distance_next_stop.mul_add(-2.4, f64::from(stop_points_distances[iii])) < 1750.0)
                {}
                else {
                    continue;
                }

                if !are_stops_in_cadastral_place(convert_to_hashable(locations_groups[min_stop_name_difference_index].1.clone()), stop_points[iii].0.clone())
                    || if_other_stop_have_same_location(
                        &mut stop_points,
                        iii,
                        locations_groups[min_stop_name_difference_index].0,
                        &locations_groups[min_stop_name_difference_index].1[0].1,
                    )
                {
                    continue;
                }

                // println!("{:#?}", locations_groups[min_stop_name_difference_index]);
                stop_points[iii].1 = locations_groups[min_stop_name_difference_index].0;
                stop_points_name_openstreetmap[iii].clone_from(&locations_groups[min_stop_name_difference_index].1[0].1);
            }
            // debug!("{stop_points:#?}");

            let temp_stops_names_forbidden_for_exact_match = find_forbidden_stops(&mut stop_points, &stop_points_name_openstreetmap, &stop_points_distances, errors_in_distances);
            let mut should_continue = false;
            for stop_name in temp_stops_names_forbidden_for_exact_match {
                if !stops_names_forbidden_for_exact_match.contains(&stop_name) {
                    stops_names_forbidden_for_exact_match.push(stop_name);
                    should_continue = true;
                }
            }
            if should_continue {
                continue 'outer_outer_outer;
            }

            for (record, ref mut stop_possible_locations) in &mut stops_records {
                for stop_point in &stop_points {
                    if stop_point.0 != record.stop_name {
                        continue;
                    }
                    assert!(!stop_point.1.latitude().is_nan());
                    assert!(!stop_point.1.longitude().is_nan());
                    // assert!(stop_point.1.latitude() != 0.0);
                    // assert!(stop_point.1.longitude() != 0.0);
                    if stop_point.1.latitude() == 0.0 || stop_point.1.longitude() == 0.0 {
                        continue;
                    }

                    if record.stop_lat == "0" || record.stop_lat.is_empty() || record.stop_lon == "0" || record.stop_lon.is_empty() {
                        record.stop_lat = stop_point.1.latitude().to_string();
                        record.stop_lon = stop_point.1.longitude().to_string();
                    } else if record.stop_lat != stop_point.1.latitude().to_string() || record.stop_lon != stop_point.1.longitude().to_string() {
                        if stop_possible_locations.is_empty() {
                            stop_possible_locations.push(Location::new(record.stop_lat.parse::<f64>().unwrap(), record.stop_lon.parse().unwrap()));
                        }
                        if stop_possible_locations.contains(&stop_point.1) {
                            continue;
                        }
                        stop_possible_locations.push(stop_point.1);
                        let center = Location::center(&stop_possible_locations.iter().collect::<Vec<&Location>>());
                        debug!("stop_possible_locations: {center:?}\n{record:#?}\n{stop_point:?}\n{stop_possible_locations:#?}");
                        record.stop_lat = center.latitude().to_string();
                        record.stop_lon = center.longitude().to_string();
                    }
                    debug!("record: {record:#?}");
                }
            }
        }

        if max_stop_points_distance == 0 && does_stop_points_distance_exists {
            // println!("breaked2: {i:?}");
        }

        if max_stop_points_distance == 0 && !does_stop_points_distance_exists {
            // println!("breaked: {i:?}");
            break;
        }

        if max_stop_points_distance != 0 {
            let mut stops_record_converted: Vec<(Location, String)> = Vec::new();
            for (stop, _) in &stops_records {
                if stop.stop_lat == "0" || stop.stop_lon == "0" {
                    continue;
                }
                stops_record_converted.push((Location::new(stop.stop_lat.parse::<f64>().unwrap(), stop.stop_lon.parse::<f64>().unwrap()), stop.stop_name.clone()));
            }
            let mut grouped_stops = group_locations_(stops_record_converted, Distance::from_meters(max_stop_points_distance * 5));
            grouped_stops.sort_by(|a, b| b.1.len().cmp(&a.1.len()));
            if !grouped_stops.is_empty() && grouped_stops[0].1.len() > grouped_stops.iter().skip(1).map(|(_, vec)| vec.len()).sum::<usize>() + 1 {
                for (stop, _) in &mut stops_records {
                    if !grouped_stops[0].1.iter().map(|x| x.1.clone()).any(|x| x == stop.stop_name) {
                        stop.stop_lat = "0".to_owned();
                        stop.stop_lon = "0".to_owned();
                    }
                }
            }
        }

        let mut buffer = Vec::new();
        let mut wtr = WriterBuilder::new().quote_style(QuoteStyle::Always).from_writer(&mut buffer);
        for (stop, _) in stops_records {
            wtr.serialize(stop).unwrap();
        }

        wtr.flush().unwrap();
        drop(wtr);
        fs::write(format!("{}/stops.txt", i.to_str().unwrap()), buffer).unwrap();
        break;
    }

    dbg!(&stops_names_forbidden_for_exact_match);
}

fn find_min_stop_name_difference_index(locations_groups: &[(Location, Vec<(Location, String)>)], stop_name: &str) -> usize {
    let mut min_distance = usize::MAX;
    let mut min_distance_index = 0;

    for (iiii, item) in locations_groups.iter().enumerate() {
        // let distance = levenshtein(&item.1[0].1, &stop_name);
        let distance = stop_names_distance(&item.1[0].1, &stop_name);

        if distance < min_distance {
            min_distance = distance;
            min_distance_index = iiii;
        }
    }

    min_distance_index
}

fn find_consecutive_duplicates(strings: &Vec<usize>) -> Vec<usize> {
    let mut consecutive_duplicates = Vec::new();

    for i in 0..strings.len().saturating_sub(1) {
        if strings[i] == strings[i + 1] {
            assert!(!(i == 0 || i == strings.len().saturating_sub(1) - 1), "{strings:?}");

            consecutive_duplicates.push(i);
        }
    }
    consecutive_duplicates
}

fn find_forbidden_stops(stop_points: &mut [(String, Location)], stop_points_name_openstreetmap: &[String], stop_points_distances: &[i32], errors_in_distances: Vec<(String, String)>) -> Vec<String> {
    let mut stop_points_distance = 0;
    let mut last_stop_point_with_location = None;
    let mut number_of_stop_points_between = 0;
    let mut possible_forbidden_stops: Vec<usize> = Vec::new();
    let mut stop_points_with_location = Vec::new();
    for iiii in 0..stop_points.len() {
        if last_stop_point_with_location.is_none() && stop_points[iiii].1.latitude() != 0.0 && iiii != stop_points.len() - 1 {
            if stop_points_name_openstreetmap[iiii].is_empty() {
                println!();
                panic!();
            }
            last_stop_point_with_location = Some(iiii);
            stop_points_distance = stop_points_distances[iiii];
            number_of_stop_points_between = 0;
            stop_points_with_location.push((stop_points[iiii].0.clone(), iiii));
        } else if stop_points[iiii].1.latitude() == 0.0 && iiii != stop_points.len() - 1 {
            stop_points_distance += stop_points_distances[iiii];
            number_of_stop_points_between += 1;
        } else if stop_points[iiii].1.latitude() != 0.0 && last_stop_point_with_location.is_some() {
            stop_points_with_location.push((stop_points[iiii].0.clone(), iiii));
            let distance: f64 = stop_points[iiii].1.distance_to(&stop_points[last_stop_point_with_location.unwrap()].1).unwrap().meters();
            if distance - f64::from(stop_points_distance) > 1500.0 || distance.mul_add(-2.4, f64::from(stop_points_distance)) > (1750.0 * (1.0 + f64::from(number_of_stop_points_between) / 2.0)) {
                if stop_points_name_openstreetmap[last_stop_point_with_location.unwrap()] != stop_points_name_openstreetmap[iiii] {
                    if !errors_in_distances
                        .iter()
                        .any(|x| x.0 == stop_points[last_stop_point_with_location.unwrap()].0 && x.1 == stop_points[iiii].0)
                    {
                        possible_forbidden_stops.push(last_stop_point_with_location.unwrap());
                        possible_forbidden_stops.push(iiii);
                    }
                }
            }
            // assert!(last_stop_point_with_location.0 != stop_points[iiii].0);
            if iiii != stop_points.len() - 1 {
                if stop_points_name_openstreetmap[iiii].is_empty() {
                    println!();
                    panic!(
                        "stop_points_name_openstreetmap: {stop_points_name_openstreetmap:?},stop_points_name_openstreetmap[iiii]: {:?}",
                        stop_points_name_openstreetmap[iiii]
                    );
                }
                last_stop_point_with_location = Some(iiii);
                stop_points_distance = stop_points_distances[iiii];
                number_of_stop_points_between = 0;
            }
        }
    }
    assert!(possible_forbidden_stops.len() % 2 == 0);
    let consecutives_duplicates = find_consecutive_duplicates(&possible_forbidden_stops);

    let mut stops_names_forbidden_for_exact_match = Vec::new();
    let mut should_continue = false;
    if !consecutives_duplicates.is_empty() && stop_points_with_location.len() > consecutives_duplicates.len() {
        for consecutives_duplicate in &consecutives_duplicates {
            if !stops_names_forbidden_for_exact_match.contains(&stop_points_name_openstreetmap[possible_forbidden_stops[*consecutives_duplicate]]) {
                let distance: f64 = stop_points[possible_forbidden_stops[consecutives_duplicate - 1]]
                    .1
                    .distance_to(&stop_points[possible_forbidden_stops[consecutives_duplicate + 2]].1)
                    .unwrap()
                    .meters();
                if distance - f64::from(stop_points_distance) > 1500.0 || distance.mul_add(-2.4, f64::from(stop_points_distance)) > (1750.0 * (1.0 + f64::from(number_of_stop_points_between) / 2.0)) {
                } else {
                    if consecutives_duplicates.contains(&(consecutives_duplicate - 1)) || consecutives_duplicates.contains(&(consecutives_duplicate + 2)) {
                    } else {
                        stops_names_forbidden_for_exact_match.push(stop_points_name_openstreetmap[possible_forbidden_stops[*consecutives_duplicate]].clone());
                        should_continue = true;
                        stop_points[possible_forbidden_stops[*consecutives_duplicate]].1 = Location::new(0.0, 0.0);
                    }
                }
            }
        }
    }
    if !possible_forbidden_stops.is_empty() && stop_points_with_location.iter().map(|v| v.0.clone()).collect::<HashSet<_>>().len() > possible_forbidden_stops.len() + 1 {
        if possible_forbidden_stops[0] == stop_points_with_location[0].1
            && (possible_forbidden_stops.len() < 4 || possible_forbidden_stops[1] != possible_forbidden_stops[2])
            && !stops_names_forbidden_for_exact_match.contains(&stop_points_name_openstreetmap[possible_forbidden_stops[0]])
        {
            if !are_stops_in_good_distance(0, 3, stop_points, &stop_points_with_location, stop_points_distances)
                && stop_points_with_location[0].0 != stop_points_with_location[3].0
                && stop_points_with_location[1].0 != stop_points_with_location[2].0
            {
                stops_names_forbidden_for_exact_match.push(stop_points_name_openstreetmap[possible_forbidden_stops[0]].clone());
                should_continue = true;
            }
        }
        if *possible_forbidden_stops.last().unwrap() == stop_points_with_location.last().unwrap().1
            && (possible_forbidden_stops.len() < 4 || possible_forbidden_stops[possible_forbidden_stops.len() - 2] != possible_forbidden_stops[possible_forbidden_stops.len() - 3])
            && !stops_names_forbidden_for_exact_match.contains(&stop_points_name_openstreetmap[*possible_forbidden_stops.last().unwrap()])
        {
            if !are_stops_in_good_distance(
                stop_points_with_location.len() - 1 - 2,
                stop_points_with_location.len() - 1,
                stop_points,
                &stop_points_with_location,
                stop_points_distances,
            ) {
                if !are_stops_in_good_distance(
                    stop_points_with_location.len() - 1 - 3,
                    stop_points_with_location.len() - 1,
                    stop_points,
                    &stop_points_with_location,
                    stop_points_distances,
                ) && stop_points_with_location[stop_points_with_location.len() - 1 - 3].0 != stop_points_with_location[stop_points_with_location.len() - 1].0
                    && stop_points_with_location[stop_points_with_location.len() - 1 - 3].0 != stop_points_with_location[stop_points_with_location.len() - 1 - 1].0
                    && stop_points_with_location[stop_points_with_location.len() - 1 - 3].0 != stop_points_with_location[stop_points_with_location.len() - 1 - 2].0
                    && !consecutives_duplicates
                        .iter()
                        .any(|x| possible_forbidden_stops[*x] == stop_points_with_location[stop_points_with_location.len() - 1 - 3].1)
                {
                    if stop_points_with_location[stop_points_with_location.len() - 1 - 2].0 != stop_points_with_location[stop_points_with_location.len() - 1 - 1].0
                        && stop_points[stop_points_with_location[stop_points_with_location.len() - 1 - 2].1].1 == stop_points[stop_points_with_location[stop_points_with_location.len() - 1 - 1].1].1
                    {
                    } else {
                        stops_names_forbidden_for_exact_match.push(stop_points_name_openstreetmap[*possible_forbidden_stops.last().unwrap()].clone());
                        should_continue = true;
                    }
                }
            }
        }
    }
    if possible_forbidden_stops.len() > 3 {
        // assert!(should_continue, "{possible_forbidden_stops:?} {stop_points_with_location:?}");
        if !should_continue {
            println!("{possible_forbidden_stops:?} {stop_points_with_location:?}");
        }
    }

    println!("forbidden_stops: {stops_names_forbidden_for_exact_match:#?}");
    println!("end find_forbidden_stops");
    if should_continue {
        dbg!(&stops_names_forbidden_for_exact_match);
        return stops_names_forbidden_for_exact_match;
    }

    Vec::new()
}

fn get_stop_points_distance(mut possible_forbidden_stops_1: usize, mut possible_forbidden_stops_2: usize, stop_points_distances: &[i32]) -> i32 {
    if possible_forbidden_stops_1 > possible_forbidden_stops_2 {
        let temp = possible_forbidden_stops_2;
        possible_forbidden_stops_2 = possible_forbidden_stops_1;
        possible_forbidden_stops_1 = temp;
    }
    let mut distance = 0;
    for i in possible_forbidden_stops_1..possible_forbidden_stops_2 {
        distance += stop_points_distances[i];
    }
    return distance;
}

fn are_stops_in_good_distance(
    stop_point_1_index: usize,
    stop_point_2_index: usize,
    stop_points: &[(String, Location)],
    stop_points_with_location: &Vec<(String, usize)>,
    stop_points_distances: &[i32],
) -> bool {
    let distance: f64 = stop_points[stop_points_with_location[stop_point_1_index].1]
        .1
        .distance_to(&stop_points[stop_points_with_location[stop_point_2_index].1].1)
        .unwrap()
        .meters();
    if distance
        - f64::from(get_stop_points_distance(
            stop_points_with_location[stop_point_1_index].1,
            stop_points_with_location[stop_point_2_index].1,
            stop_points_distances,
        ))
        > 1500.0
        || distance.mul_add(
            -2.4,
            f64::from(get_stop_points_distance(
                stop_points_with_location[stop_point_1_index].1,
                stop_points_with_location[stop_point_2_index].1,
                stop_points_distances,
            )),
        ) > (1750.0 * (1.0 + (stop_points_with_location[stop_point_2_index].1 - stop_points_with_location[stop_point_1_index].1 - 1) as f64 / 2.0))
    {
        println!(
            "not in good distance {:?} {:?} {} {}",
            stop_points[stop_points_with_location[stop_point_1_index].1],
            stop_points[stop_points_with_location[stop_point_2_index].1],
            distance,
            get_stop_points_distance(stop_points_with_location[stop_point_1_index].1, stop_points_with_location[stop_point_2_index].1, stop_points_distances,)
        );
        return false;
    }

    println!(
        "in good distance {:?} {:?} {} {}",
        stop_points[stop_points_with_location[stop_point_1_index].1],
        stop_points[stop_points_with_location[stop_point_2_index].1],
        distance,
        get_stop_points_distance(stop_points_with_location[stop_point_1_index].1, stop_points_with_location[stop_point_2_index].1, stop_points_distances,)
    );
    return true;
}

fn parse_time(time_str: &str) -> f64 {
    let parts: Vec<&str> = time_str.split(':').collect();
    if parts.len() != 3 {
        panic!("Invalid time format");
    }

    let hours: u32 = parts[0].parse().unwrap();
    let minutes: u32 = parts[1].parse().unwrap();
    let seconds: u32 = parts[2].parse().unwrap();

    assert!(seconds == 0);

    (hours * 3600 + minutes * 60 + seconds) as f64
}

fn normalize_stop_name(input: &str) -> String {
    let s = regex_whitespace().replace_all(input, " ").to_string();
    for (index, i) in regex_brackets().find_iter(&s).enumerate() {
        assert!(index == 0);
        assert!(i.len() == 4);
    }
    let s = regex_brackets().replace_all(&s, "").to_string();

    let s = regex_number().replace_all(&s, "").to_string();

    let s = s.replace(',', " ").replace('.', ". ");

    let s = regex_whitespace().replace_all(&s, " ").to_string();

    s.to_lowercase().trim().to_owned()
}

fn stop_names_distance(filter: &str, ii: &str) -> usize {
    let mut a = &normalize_stop_name(filter);
    let mut b = &normalize_stop_name(ii);

    if b.contains('.') {
        std::mem::swap(&mut a, &mut b);
    }

    let a_ = a.split(' ').collect::<Vec<_>>();
    let b_ = b.split(' ').collect::<Vec<_>>();

    let mut count = 0;
    let mut count_only_short = 0;
    if a.contains('.') {
        for i in &a_ {
            if i.chars().count() < 3 && !i.contains('.') {
                if i.contains('.') && b_.iter().any(|x| x.starts_with(&i[..i.len() - 1])) || b_.contains(i) {
                    count_only_short += 1;
                }
                continue;
            }
            if i.contains('.') && b_.iter().any(|x| x.starts_with(&i[..i.len() - 1])) || b_.contains(i) {
                count += 1;
            }
        }
    } else {
        for i in &a_ {
            if i.chars().count() < 3 {
                if b_.contains(i) {
                    count_only_short += 1;
                }
                continue;
            }
            if b_.contains(i) {
                count += 1;
            }
        }
    }
    if count == 0 {
        if filter == ii {
            panic!("{filter} {ii} {a_:#?} {b_:#?} {count}");
        }
        return 1_000_000;
    }
    if count >= 4 || (a_.len() == b_.len() && a_.len() == count) || (a_.len() == b_.len() && a_.len() == count + count_only_short) {
        return 0;
    }
    if filter == ii {
        panic!("{filter} {ii} {a_:#?} {b_:#?} {count}");
    }
    if count >= 3 {
        assert!(count >= 1, "{a_:#?} {b_:#?}");
        return 1;
    }

    levenshtein(a, b)
}

fn regex_whitespace() -> &'static Regex {
    static RE: OnceLock<Regex> = OnceLock::new();
    RE.get_or_init(|| Regex::new(r"\s+").unwrap())
}

fn regex_brackets() -> &'static Regex {
    static RE: OnceLock<Regex> = OnceLock::new();
    RE.get_or_init(|| Regex::new(r"\[.*?\]").unwrap())
}

fn regex_number() -> &'static Regex {
    static RE: OnceLock<Regex> = OnceLock::new();
    RE.get_or_init(|| Regex::new(r"\d+(,|\.)\d+").unwrap())
}
