#[derive(Serialize, Deserialize, Debug)]
struct Document {
    id: i64,
    name: String,
    _geo: String,
}

use core::time;
use geoutils::{Distance, Location};
use meilisearch_sdk::client::*;
use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::env;
use std::io::BufReader;
use std::{fs::File, io::prelude::*, thread::sleep};

#[tokio::main(flavor = "current_thread")]
async fn main() {
    const INDEX_NAME: &str = "bus_tram_stops";

    let args: Vec<String> = env::args().collect();
    println!("{args:#?}");
    let client = Client::new("http://127.0.0.1:7700", Some(args[2].clone())).unwrap();

    let mut content: Vec<Document> = Vec::new();

    let file = File::open("bus_tram_stops.json").unwrap();
    let reader = BufReader::new(file);
    let value: Value = serde_json::from_reader(reader).unwrap();
    let mut number_of_nodes_without_name_or_official_name = 0;

    if let Some(numbers) = value["elements"].as_array() {
        for (index, record) in numbers.iter().enumerate() {
            let name = match &record["tags"]["official_name"] {
                Value::String(v) => v,
                _ => {
                    match &record["tags"]["name"] {
                        Value::String(v) => v,
                        _ => {
                            // println!("{:#?} {} {:#?}", record["tags"]["name"], index, record);
                            number_of_nodes_without_name_or_official_name += 1;
                            continue;
                        }
                    }
                }
            };
            let coordinates = if record["lat"].is_f64() {
                assert!(record["lon"].is_f64());
                (record["lat"].to_string(), record["lon"].to_string())
            } else {
                assert!(!record["lon"].is_f64());
                assert!(record["bounds"]["minlat"].is_f64());
                assert!(record["bounds"]["minlon"].is_f64());
                assert!(record["bounds"]["maxlat"].is_f64());
                assert!(record["bounds"]["maxlon"].is_f64());
                let location = Location::center(&[
                    &Location::new(record["bounds"]["minlat"].as_f64().unwrap(), record["bounds"]["minlon"].as_f64().unwrap()),
                    &Location::new(record["bounds"]["maxlat"].as_f64().unwrap(), record["bounds"]["maxlon"].as_f64().unwrap()),
                ]);
                (location.latitude().to_string(), location.longitude().to_string())
            };
            // assert!(record["lat"].is_f64());
            // assert!(record["lon"].is_f64());
            content.push(Document {
                id: index as i64,
                name: name.to_string(),
                _geo: coordinates.0 + "," + &coordinates.1,
            });
        }
    }
    println!("number of bus tram stops that were excluded because they don't have name or official_name: {number_of_nodes_without_name_or_official_name}");

    let abc = client.index(INDEX_NAME).add_documents(&content, Some("id")).await.unwrap();
    println!("{abc:#?}");

    loop {
        sleep(time::Duration::from_millis(4000));

        let abc2 = client.get_task(&abc).await.unwrap();
        if abc2.is_success() {
            println!("{abc2:#?}");
            break;
        }
    }

    let searchable_attributes = ["name"];

    let task = client.index(INDEX_NAME).set_searchable_attributes(&searchable_attributes).await.unwrap();

    loop {
        sleep(time::Duration::from_millis(2000));

        let abc2 = client.get_task(&task).await.unwrap();
        if abc2.is_success() {
            println!("{abc2:#?}");
            break;
        }
    }

    println!("completed");
}
