use criterion::{black_box, criterion_group, criterion_main, Criterion};
use geoutils::Location;
use rand::{rng, Rng};

fn generate_test_data(size: usize) -> Vec<(Location, Vec<(Location, String)>)> {
    let mut rng = rng();

    let mut groups = Vec::with_capacity(size);
    for _ in 0..size {
        let center_lat = rng.random_range(-80.0..80.0);
        let center_lon = rng.random_range(-170.0..170.0);
        let center = Location::new(center_lat, center_lon);

        // Generate 1-5 nearby points for each group
        let nearby_count = rng.random_range(1..=5);
        let nearby_points = (0..nearby_count)
            .map(|_| {
                // Add small random offsets (roughly within 1km)
                let lat_offset = rng.random_range(-0.01..0.01);
                let lon_offset = rng.random_range(-0.01..0.01);
                let nearby_loc = Location::new(
                    center_lat + lat_offset,
                    center_lon + lon_offset
                );
                (nearby_loc, format!("point_{}", rng.random_range(0..1000)))
            })
            .collect();

        groups.push((center, nearby_points));
    }
    groups
}

fn find_min_distance_loop(locations_groups: &[(Location, Vec<(Location, String)>)], center: &Location) -> usize {
    let mut min_distance = f64::MAX;
    let mut min_distance_index = 0;
    for (iiii, item) in locations_groups.iter().enumerate() {
        let distance = match item.0.distance_to(center) {
            Ok(v) => v.meters(),
            Err(e) => panic!("error: {e}, {:#?} {:#?}", item.0, center)
        };
        if distance < min_distance {
            min_distance = distance;
            min_distance_index = iiii;
        }
    }
    min_distance_index
}

fn find_min_distance_iterator(locations_groups: &[(Location, Vec<(Location, String)>)], center: &Location) -> usize {
    locations_groups
        .iter()
        .enumerate()
        .map(|(idx, item)| (idx, item.0.distance_to(center).unwrap().meters()))
        .min_by(|(_, dist_a), (_, dist_b)| dist_a.partial_cmp(dist_b).unwrap())
        .unwrap()
        .0
}

fn criterion_benchmark(c: &mut Criterion) {
    let test_sizes = [10, 100, 1000, 10000];

    for size in test_sizes {
        let locations_groups = generate_test_data(size);
        let center = Location::new(0.0, 0.0);

        let mut group = c.benchmark_group(format!("location_search_{size}"));

        group.bench_function("loop_version", |b| {
            b.iter(|| find_min_distance_loop(black_box(&locations_groups), black_box(&center)));
        });

        group.bench_function("iterator_version", |b| {
            b.iter(|| find_min_distance_iterator(black_box(&locations_groups), black_box(&center)));
        });

        group.finish();
    }
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);
